'use client'

import { useState, useEffect } from 'react'
import { useParams, useRouter } from 'next/navigation'
import Link from 'next/link'
import { multiTripIdentity } from '@/lib/multiTripIdentity'
import ExpenseModal from '@/components/ExpenseModal'
import TripTabs from '@/components/TripTabs'
import CopyLink from '@/components/CopyLink'
import ExportButton from '@/components/ExportButton'
import DarkModeToggle from '@/components/DarkModeToggle'

export default function TripPage() {
  const params = useParams()
  const router = useRouter()
  const [trip, setTrip] = useState<any>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const [userIdentity, setUserIdentity] = useState<any>(null)
  const [showExpenseModal, setShowExpenseModal] = useState(false)
  const [refreshTrigger, setRefreshTrigger] = useState(0)
  const [activeTab, setActiveTab] = useState<'expenses' | 'analytics'>('expenses')
  const [needsPassword, setNeedsPassword] = useState(false)
  const [password, setPassword] = useState('')
  const [authenticating, setAuthenticating] = useState(false)

  useEffect(() => {
    // If no redirect data, proceed with normal flow
    const tripId = params.id as string
    const identity = multiTripIdentity.getTripIdentity(tripId)

    if (!identity) {
      // User not identified for this trip, redirect to join page
      router.push('/join')
      return
    }

    setUserIdentity(identity)

    // Update last accessed time
    multiTripIdentity.updateLastAccessed(tripId)

    // Load trip data
    loadTripData()
  }, [params.id, router])

  const loadTripData = async () => {
    const tripId = parseInt(params.id as string)

    try {
      // Use the API instance with configured baseURL
      const { tripApi, getTripToken } = await import('@/lib/api')
      const token = getTripToken(tripId)

      // IMPORTANT: Wait a moment for token to be available if we just navigated
      if (!token) {
        console.log('No token available, waiting briefly for token to appear...')
        await new Promise(resolve => setTimeout(resolve, 100))

        // Check again after brief wait
        const tokenAfterWait = getTripToken(tripId)
        if (tokenAfterWait) {
          console.log('Token found after brief wait')
        }
      }

      const data = await tripApi.getById(tripId)
      setTrip(data)
      setLoading(false)
    } catch (error: any) {
      // Check for 401 errors
      if (error.message?.includes('401') || error.status === 401 || error.response?.status === 401) {
        console.log('401 detected - showing password prompt')
        setError('This trip is password protected. Please enter the password.')
        setNeedsPassword(true)
        setLoading(false)
        return
      }

      // If we get here, it's a real error
      setError('Failed to load trip data')
      console.error('Final error:', error)
      setLoading(false)
    }
  }

  const handlePasswordSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!password.trim()) {
      setError('Please enter a password')
      return
    }

    setAuthenticating(true)
    setError('')

    try {
      const tripId = parseInt(params.id as string)
      const { authenticateTrip } = await import('@/lib/trip-auth')
      await authenticateTrip(tripId, password.trim())
      
      // Now try to load trip data again with authentication
      setNeedsPassword(false)
      setPassword('')
      setLoading(true)
      await loadTripData()
    } catch (err: any) {
      setError(err.message || 'Incorrect password. Please try again.')
      setPassword('')
    } finally {
      setAuthenticating(false)
    }
  }

  const handleAddExpense = () => {
    setShowExpenseModal(true)
  }

  const handleExpenseCreated = (expense: any) => {
    setRefreshTrigger(prev => prev + 1) // Trigger expense list refresh
  }

  const handleAddParticipant = async () => {
    const name = prompt('Enter participant name:')
    if (!name || !name.trim()) return

    try {
      // Use the API instance with configured baseURL
      const { participantApi } = await import('@/lib/api')
      const tripId = parseInt(params.id as string)
      await participantApi.create(tripId, {
        name: name.trim(),
      })

      // Reload trip data
      await loadTripData()
    } catch (error) {
      alert('Failed to add participant. Please try again.')
      console.error(error)
    }
  }

  const handleLeaveTrip = () => {
    if (confirm('Are you sure you want to leave this trip? This will remove your access to this trip.')) {
      const tripId = params.id as string
      multiTripIdentity.removeTripIdentity(tripId)
      router.push('/dashboard')
    }
  }

  // Show password prompt if needed
  if (needsPassword) {
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="w-full max-w-md space-y-6">
          <div className="text-center space-y-2">
            <h1 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Password Required</h1>
            <p className="text-gray-600 dark:text-gray-400">This trip is password protected</p>
          </div>

          <form onSubmit={handlePasswordSubmit} className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
            {error && (
              <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg text-sm">
                {error}
              </div>
            )}

            <div>
              <label htmlFor="trip-password" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Trip Password
              </label>
              <input
                id="trip-password"
                type="password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                placeholder="Enter trip password"
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                required
                autoFocus
              />
            </div>

            <div className="flex space-x-3">
              <button
                type="button"
                onClick={() => router.push('/dashboard')}
                className="flex-1 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-2 px-4 rounded-lg font-medium hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
              >
                Back
              </button>
              <button
                type="submit"
                disabled={authenticating || !password.trim()}
                className="flex-1 bg-blue-600 dark:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium hover:bg-blue-700 dark:hover:bg-blue-600 disabled:bg-gray-400 disabled:cursor-not-allowed transition-colors"
              >
                {authenticating ? 'Authenticating...' : 'Continue'}
              </button>
            </div>
          </form>
        </div>
      </main>
    )
  }

  if (loading) {
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600 dark:text-gray-400">Loading trip...</p>
        </div>
      </main>
    )
  }

  if (error || !trip) {
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="text-center space-y-4">
          <h1 className="text-2xl font-bold text-red-600 dark:text-red-400">Error</h1>
          <p className="text-gray-600 dark:text-gray-400">{error || 'Trip not found'}</p>
          <button
            onClick={() => router.push('/')}
            className="bg-blue-600 dark:bg-blue-700 text-white py-2 px-4 rounded-lg hover:bg-blue-700 dark:hover:bg-blue-600"
          >
            Back to Home
          </button>
        </div>
      </main>
    )
  }

  return (
    <main className="min-h-screen bg-gray-50 dark:bg-gray-900 flex flex-col">
      {/* Header */}
      <div className="bg-white dark:bg-gray-800 border-b dark:border-gray-700 sticky top-0 z-10">
        <div className="max-w-md mx-auto p-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-xl font-bold text-gray-900 dark:text-gray-100">{trip.name}</h1>
              <p className="text-sm text-gray-500 dark:text-gray-400">Code: {trip.share_code}</p>
            </div>
            <div className="flex items-center space-x-2">
              <DarkModeToggle />
              <ExportButton
                tripId={trip.id}
                shareCode={trip.share_code}
              />
              <button
                onClick={() => router.push('/dashboard')}
                className="text-gray-400 hover:text-gray-600 dark:text-gray-500 dark:hover:text-gray-300"
                title="View all trips"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
              <button
                onClick={handleLeaveTrip}
                className="text-gray-400 hover:text-gray-600 dark:text-gray-500 dark:hover:text-gray-300"
                title="Leave trip"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" />
                </svg>
              </button>
            </div>
          </div>
        </div>
      </div>

      <div className="flex-1 max-w-md mx-auto p-4 pb-20 space-y-4">
        {/* User Welcome */}
        <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 p-4 rounded-lg">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-blue-600 dark:text-blue-400">Welcome back,</p>
              <p className="font-semibold text-blue-900 dark:text-blue-100">{userIdentity?.participantName}</p>
            </div>
            <div className="text-right">
              <p className="text-xs text-blue-500 dark:text-blue-400">Trip Currency</p>
              <p className="font-bold text-blue-900 dark:text-blue-100">{trip.currency_code}</p>
            </div>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="grid grid-cols-2 gap-4">
          <button
            onClick={handleAddExpense}
            className="bg-blue-600 dark:bg-blue-700 text-white p-4 rounded-lg hover:bg-blue-700 dark:hover:bg-blue-600 transition-colors"
          >
            <svg className="w-6 h-6 mx-auto mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
            </svg>
            <p className="text-sm font-medium">Add Expense</p>
          </button>

          <button
            onClick={handleAddParticipant}
            className="bg-green-600 dark:bg-green-700 text-white p-4 rounded-lg hover:bg-green-700 dark:hover:bg-green-600 transition-colors"
          >
            <svg className="w-6 h-6 mx-auto mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M18 9v3m0 0v3m0-3h3m-3 0h-3m-2-5a4 4 0 11-8 0 4 4 0 018 0zM3 20a6 6 0 0112 0v1H3v-1z" />
            </svg>
            <p className="text-sm font-medium">Add Person</p>
          </button>
        </div>

        {/* Expense Management */}
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <h2 className="font-semibold text-gray-900 dark:text-gray-100">Expense Management</h2>
            {activeTab === 'expenses' && (
              <button
                onClick={handleAddExpense}
                className="text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 text-sm font-medium"
              >
                Add Expense
              </button>
            )}
          </div>
          <TripTabs
            activeTab={activeTab}
            onTabChange={setActiveTab}
            tripId={parseInt(params.id as string)}
            tripCurrency={trip.currency_code}
            participants={trip.participants || []}
            refreshTrigger={refreshTrigger}
            onExpenseUpdated={() => setRefreshTrigger(prev => prev + 1)}
          />
        </div>

        {/* Participants */}
        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
          <h2 className="font-semibold text-gray-900 dark:text-gray-100 mb-3">Participants ({trip.participants?.length || 0})</h2>
          <div className="space-y-2">
            {(trip.participants || []).map((participant: any) => (
              <div key={participant.id} className="flex items-center justify-between p-2">
                <div className="flex items-center space-x-3">
                  <div className="w-8 h-8 bg-blue-100 dark:bg-blue-900/30 rounded-full flex items-center justify-center">
                    <span className="text-sm font-medium text-blue-700 dark:text-blue-300">
                      {participant.name.charAt(0).toUpperCase()}
                    </span>
                  </div>
                  <div>
                    <p className="font-medium text-gray-900 dark:text-gray-100">{participant.name}</p>
                    {participant.is_creator && (
                      <p className="text-xs text-gray-500 dark:text-gray-400">Trip Creator</p>
                    )}
                  </div>
                </div>
                {participant.name === userIdentity?.participantName && (
                  <span className="text-xs bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 px-2 py-1 rounded">You</span>
                )}
              </div>
            ))}
          </div>
        </div>

        {/* Share and Export Actions */}
        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
          <h2 className="font-semibold text-gray-900 dark:text-gray-100 mb-3">Share & Export</h2>
          <div className="grid grid-cols-2 gap-3">
            <CopyLink
              shareCode={trip.share_code}
            />
            <ExportButton
              tripId={trip.id}
              shareCode={trip.share_code}
            />
          </div>
        </div>
      </div>

      {/* Bottom Navigation */}
      <div className="bg-white dark:bg-gray-800 border-t dark:border-gray-700 sticky bottom-0 z-10">
        <div className="max-w-md mx-auto">
          <div className="grid grid-cols-3 py-2">
            <Link
              href="/dashboard"
              className="flex flex-col items-center py-2 text-blue-600 dark:text-blue-400"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
              </svg>
              <span className="text-xs mt-1">My Trips</span>
            </Link>

            <button
              onClick={handleAddExpense}
              className="flex flex-col items-center py-2 text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
              </svg>
              <span className="text-xs mt-1">Add Expense</span>
            </button>

            <button
              onClick={handleAddParticipant}
              className="flex flex-col items-center py-2 text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M18 9v3m0 0v3m0-3h3m-3 0h-3m-2-5a4 4 0 11-8 0 4 4 0 018 0zM3 20a6 6 0 0112 0v1H3v-1z" />
              </svg>
              <span className="text-xs mt-1">People</span>
            </button>
          </div>
        </div>
      </div>

      {/* Expense Modal */}
      <ExpenseModal
        isOpen={showExpenseModal}
        onClose={() => setShowExpenseModal(false)}
        tripId={parseInt(params.id as string)}
        participants={trip.participants || []}
        tripCurrency={trip.currency_code}
        onExpenseCreated={handleExpenseCreated}
      />
    </main>
  )
}